/*
 * Copyright (c) 2025, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package jdk.jpackage.internal;

import static jdk.jpackage.internal.util.XmlUtils.initDocumentBuilder;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import jdk.jpackage.internal.model.DottedVersion;
import jdk.jpackage.internal.util.PListReader;
import org.xml.sax.SAXException;

/**
 * Mandatory elements of Info.plist file of app image.
 */
record AppImageInfoPListFile(String bundleIdentifier, String bundleName, String copyright,
        DottedVersion shortVersion, DottedVersion bundleVersion, String category) {

    static final class InvalidPlistFileException extends Exception {
        InvalidPlistFileException(Throwable cause) {
            super(cause);
        }

        private static final long serialVersionUID = 1L;
    }

    static AppImageInfoPListFile loadFromInfoPList(Path infoPListFile)
            throws IOException, InvalidPlistFileException, SAXException {

        final var plistReader = new PListReader(initDocumentBuilder().parse(Files.newInputStream(infoPListFile)));

        try {
            return new AppImageInfoPListFile(
                    plistReader.queryValue("CFBundleIdentifier"),
                    plistReader.queryValue("CFBundleName"),
                    plistReader.queryValue("NSHumanReadableCopyright"),
                    DottedVersion.greedy(plistReader.queryValue("CFBundleShortVersionString")),
                    DottedVersion.greedy(plistReader.queryValue("CFBundleVersion")),
                    plistReader.queryValue("LSApplicationCategoryType"));
        } catch (Exception ex) {
            throw new InvalidPlistFileException(ex);
        }
    }
}
